%{
/*
 * p/n
 * Program name:  Dump Formatter Syntax
 * Author:        Alan Ballard
 * Backup:
 * Description:
 *   This component contains the YACC grammar for the dump formatter
 *   program command language.
 */
%}

%token  COMMENT MEM FORM ADDR PRINT SHELL
%token  NUMBER REM FILENAME SYMBOL
%token  USER TTY BUF CORETAB TEXT IOQ DASD TUBE MOUNT INODE FILSTRUC
%token  PROC ACTIVE UID PID PGROUP COMMAND STACK
%token  CHAN CU UNIT
%left   '+' '-'
%left   '*' '/'

%{
#include "dump.h"
#include <stdio.h>
#include "tnode.h"

extern int    aptnum;           /* decimal numeric value from scanner */
extern int    cmdecho;          /* echo requests if true */
extern FILE *out;               /* output file */
extern char   prompt[];

%}
%%
input:
        input req '\n'          { if ($2)
                                       doreq($2, 0);
                                   if (cmdecho)
                                       fprintf(out, "\n\n");
                                   fprintf(stderr, prompt);
                                }
|       empty                   { fprintf(stderr, prompt); }
|       input '\n'              { if (cmdecho)
                                       fprintf(out, "\n\n");
                                   fprintf(stderr, prompt);
                                   $$ = 0;
                                }
;
req:

        '!'                     { $$ = tnode('!', 0, 0, 0); endshell(); }
|       '!' REM                 { $$ = tnode('!', $2, 0, 0); }
|       COMMENT                 { $$ = 0; }
|       MEM expr                { $$ = tnode(MEM, 0, $2, 0); }
|       MEM expr int            { $$ = tnode(MEM, 0, $2, $3); }
|       ADDR expr               { $$ = tnode(ADDR, 0, $2, 0); }
|       PRINT dumploc           { $$ = tnode(PRINT, 0, $2, 0); }
|       FORM
                                { strtform(); }
            formoptions         { $$ = tnode(FORM, 0, $3, 0);
                                   strtnorm();
                                }
|       error                   { synerr();  $$ = 0; }
;
expr:
        SYMBOL                  { $$ = tnode(SYMBOL, $1, 0, 0); }
|       NUMBER                  { $$ = tnode(NUMBER, $1, 0, 0); }
|       expr '+' expr           { $$ = tnode('+', 0, $1, $3); }
|       expr '-' expr           { $$ = tnode('-', 0, $1, $3); }
|       expr '*' expr           { $$ = tnode('*', 0, $1, $3); }
|       expr '/' expr           { $$ = tnode('/', 0, $1, $3); }
|       '(' expr ')'            { $$ = $2; }
;
int:
        NUMBER                  { $$ = aptnum;     /* decimal value */ }
;
dumploc:
        filename                { /* not implemented yet */ }
|       empty                   { /* not implemented yet */ }
;
addr:
        tablename '{'
                                { strtnorm(); }
            expr '}'           { $$ = tnode(ADDREXPR, $1, $4,  0);
                                   strtform();
                                }
;
tablename:
        arrayname
|       simplename
;
arrayname:
        PROC                    { $$ = proccode; }
|       unit
|       DASD                    { $$ = dasdcode; }
|       TUBE                    { $$ = tubecode; }
|       MOUNT                   { $$ = mountcode; }
|       INODE                   { $$ = inodecode; }
|       BUF                     { $$ = bufcode; }
|       TEXT                    { $$ = textcode; }
|       FILSTRUC                { $$ = filecode; }
|       CHAN                    { $$ = chancode; }
|       CU                      { $$ = cucode;}
;
simplename:
        USER                    { $$ = usercode; }
|       TTY                     { $$ = ttycode; }
|       ioq                     { $$ = ioqcode; }
|       ioq ACTIVE              { $$ = activecode; }
|       CORETAB                 { $$ = coretabcode; }
|       COMMAND                 { $$ = commandcode; }
|       STACK                   { $$ = stackcode; }
;
ioq:
        IOQ
;
unit:
        UNIT                    { $$ = unitcode; }
;
formoptions:
        formopt
|       formopt formoptions     { $1 -> t_right = $2; }
;
formopt:
        tablename               { $$ = tnode(TABLE, $1, 0, 0); }
|       addr
|       arrayname '[' int ']' { $$ = tnode(ARR, $1, $3, 0); }
|       unit '(' NUMBER ')'     { $$ = tnode(DEV, $3, 0, 0); }
|       procopt
;
procopt:
        UID '(' int ')'         { $$ = tnode(PROC, UID, $3, 0); }
|       PID '(' int ')'         { $$ = tnode(PROC, PID, $3, 0); }

;
filename:
                                { strtfn(); }
        FILENAME                { $$ = $2; }
;
empty:
;
%%

/*
 *  tnode allocates a command tree node and sets its fields.
 */
struct tnode *tnode(o, v, l, r)
int o, v;
struct tnode *l, *r;
{
	register struct tnode *p;

	if ((p = calloc(1, sizeof(struct tnode))) == NULL)
                error(2, "out of memory");
	p->t_op = o;
	p->t_val = v;
	p->t_left = l;
	p->t_right = r;
	return(p);
}

/*
 *  yyerror calls are ignored.  synerr is called from the grammar
 *  This is because I can't get yyerror to work reasonably.
 */
yyerror(str)
char    *str;
{
}

/*
 *  synerr does trivial error recovery.  It reports a syntax
 *  error, then flushes to a newline.
 */
synerr()
{
	extern char yytext[];

        error(0, "syntax error: %s", yytext);
        if (yychar != '\n') {
                gobble();
                yyclearin;
        }
        yyerrok;
}

/*
 *  First level of tree evaluation.   This part is a substitute for
 *  equivalent routine in omak and dcon.
 *  The routines that are specific to dump printing are in exec.c
 */

/*
 * doreq is the basic recursive tree evaluation routine.
 *
 * parameter savflag is set if the caller does not want his tree
 * discarded; otherwise the nodes are released as the routine
 * returns.
 */
doreq(p, savflag)
struct tnode *p;
int savflag;
{
	register int v;
	register struct tnode *t1, *t2;
        struct formatentry *formp;
        int r, plen, temp, buf;

	v = p->t_val;
	t1 = p->t_left;
	t2 = p->t_right;
	r = 0;
	switch(p->t_op) {
                /*
                 *  first, the commands...
                 */
                case '!':
			if (v == NULL)
				v = "/bin/sh -i";
			system(v);
			cfree(v);
			printf("!\n");
			break;

                case FORM:
                case PRINT:
                case MEM:
                case ADDR:
                        dodump(p,savflag);
                        break;

                /*
                 *  Operands of expressions ...
                 */
		case NUMBER:
			r = v;
			break;

		case SYMBOL:
                        r = symlook(v);
                        if (r < 0)
                                error(0,"undefined symbol: %s", v);
			break;

		case '+':
		case '-':
		case '*':
                case '/':
                        r = calc(p -> t_op, t1, t2, savflag);
                        break;

		default:
                        error(1, "unknown request %x", p->t_op);
        }
	if (savflag == 0)
                cfree(p);
	return(r);
}

/*
 * calc performs the arithmetic operations.
 */
calc(op, t1, t2, savflag)
int op;
struct tnode *t1, *t2;
int savflag;
{
	register int v1, v2;

        v1 = doreq(t1, savflag);
	if (t2) v2 = doreq(t2, savflag);
	switch(op) {
		case '+':
			return(v1+v2);
		case '-':
			return(v1-v2);
		case '*':
			return(v1*v2);
                case '/':
			return(v1/v2);
		default:
			error(1, "unknown op %d", op);
        }
}

